/******************************************************************************
*                             EddeFlex All rights reserved
*******************************************************************************
*                    File: OSEK.c
*             Description: OS
*      Platform Dependent: no
*                   Notes:
******************************************************************************/


/******************************************************************************
*                              include section
******************************************************************************/
#include "Std_Types.h"
#include "rom.h"
#include "hw_memmap.h"
#include "EddeOS_Cfg.h"
#include "EddeOS.h"
#include "EddeOS_App.h"
#include "WDT.h"


/******************************************************************************
*                              MACROS
******************************************************************************/
#define NO_EVENT_ACTIVITED    0U
#define EDDEOS_TICKES_RES     1000U

/******************************************************************************
*                              Shared variables
******************************************************************************/
extern TaskObject Tasks[MAX_NUM_OF_TASKS];
extern AlarmObject Alarms[MAX_NUM_OF_ALARMS];


/******************************************************************************
*                              local functions prototypes
******************************************************************************/
static void Update_OS_Alarms(void);
static void OS_Running(void);

/******************************************************************************
*                              Local variables
******************************************************************************/
static TaskType Running_Task = MAX_NUM_OF_TASKS;
static boolean OS_No_Errors = TRUE;
static boolean Is_New_Task_Activited = TRUE;
static uint8 Is_Event_Activited = NO_EVENT_ACTIVITED;
static AlarmType ticks;

/******************************************************************************
*                              OS execution section
******************************************************************************/

/******************************************************************************
*       Function name: EddeOS_Init
*         Description: Init the OS
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void EddeOS_Init (void)
{
    uint32 EddeOS_TimerTickes;
    uint32 EddeOS_Processor_Clock;
    EddeOS_Processor_Clock = ROM_SysCtlClockGet();
    EddeOS_TimerTickes = EddeOS_Processor_Clock / EDDEOS_TICKES_RES;
    ROM_SysTickPeriodSet(EddeOS_TimerTickes);
    ROM_SysTickIntEnable();
    ROM_SysTickEnable();
}


/******************************************************************************
*       Function name: EddeOS_StartOS
*         Description: Starting the OS
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void EddeOS_StartOS ( AppModeType Mode)
{
	if(OS_NORMAL_MODE == Mode)
	{
		OS_Running();
	}
}


/******************************************************************************
*       Function name: EddeOS_ShutdownOS
*         Description: Shutdown OS
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void EddeOS_ShutdownOS ( StatusType Error)
{

}


/******************************************************************************
*       Function name: OS_Running
*         Description: OS background Task
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
static void OS_Running(void)
{
    while(OS_No_Errors)
	{
#if(OS_WDT == OS_ENABLE)
		Feed_The_Dog();
#endif
		if((TRUE == Is_New_Task_Activited) || (NO_EVENT_ACTIVITED != Is_Event_Activited))
		{
			Is_New_Task_Activited = FALSE;
			Schedule();
		}
	}
}


/******************************************************************************
*                              OS Task management section
******************************************************************************/


/******************************************************************************
*       Function name: ActivateTask
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
StatusType ActivateTask ( TaskType Task_ID )
{
	StatusType RetVal = E_OS_ID;
	if(MAX_NUM_OF_TASKS > Task_ID)
	{
		if(TASK_READY == Tasks[Task_ID].Task_State)
		{
			RetVal = E_OS_LIMIT;
		}
		else if(TASK_SUSPENDED == Tasks[Task_ID].Task_State)
		{
			RetVal = E_OS_OK;
			Tasks[Task_ID].Task_State = TASK_READY;
			Is_New_Task_Activited = TRUE;
		}
		else
		{

		}
	}
	return RetVal;
}


/******************************************************************************
*       Function name: TerminateTask
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
StatusType TerminateTask ( void )
{
	Tasks[Running_Task].Task_State = TASK_SUSPENDED;
	return E_OS_OK;
}


/******************************************************************************
*       Function name: Schedule
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
StatusType Schedule ( void )
{
	TaskType OS_Tasks;

	for(OS_Tasks = 0U; OS_Tasks < MAX_NUM_OF_TASKS; OS_Tasks++)
	{
		if(TASK_READY == Tasks[OS_Tasks].Task_State)
		{
			Running_Task = OS_Tasks;
			Tasks[OS_Tasks].Task_State = TASK_RUNNING;

#if(OS_WDT == OS_ENABLE)
		    Feed_The_Dog();
#endif
			Tasks[OS_Tasks].TaskPtr();

#if(OS_WDT == OS_ENABLE)
		    Feed_The_Dog();
#endif
		}
	}

	return E_OS_OK;
}


/******************************************************************************
*       Function name: GetTaskID
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
StatusType GetTaskID ( TaskRefType Task_ID )
{
	*Task_ID = Running_Task;
	return E_OS_OK;
}


/******************************************************************************
*       Function name: GetTaskState
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
StatusType GetTaskState ( TaskType Task_ID,TaskStateRefType State )
{
	StatusType RetVal = E_OS_ID;
	if(MAX_NUM_OF_TASKS > Task_ID)
	{
		*State = Tasks[Task_ID].Task_State;
		RetVal = E_OS_OK;
	}
	return RetVal;
}



/******************************************************************************
*                              OS Task management section
******************************************************************************/


/******************************************************************************
*       Function name: SetRelAlarm
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
StatusType SetRelAlarm( AlarmType Alarm_ID, TickType Increment, TickType Cycle )
{
	StatusType RetVal = E_OS_ID;
	
	if(MAX_NUM_OF_ALARMS > Alarm_ID)
	{
	    Alarms[Alarm_ID].cycle = Cycle;
	    Alarms[Alarm_ID].increment = Increment;
	    RetVal = E_OS_OK;
	}
	return RetVal;
}



/******************************************************************************
*       Function name: SetAbsAlarm
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
StatusType SetAbsAlarm( AlarmType Alarm_ID, TickType Start, TickType Cycle )
{
	StatusType RetVal = E_OS_ID;
	
	if(MAX_NUM_OF_ALARMS > Alarm_ID)
	{
		Alarms[Alarm_ID].cycle = Cycle;
	    Alarms[Alarm_ID].increment = Start;
	    RetVal = E_OS_OK;
	}
	return RetVal;
}


/******************************************************************************
*       Function name: GetAlarm
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
StatusType GetAlarm ( AlarmType Alarm_ID,TickRefType Tick)
{
	StatusType RetVal = E_OS_ID;
	
	if(MAX_NUM_OF_ALARMS > Alarm_ID)
	{
	    *Tick = Alarms[Alarm_ID].increment;
	    RetVal = E_OS_OK;
	}
	return RetVal;
}



/******************************************************************************
*       Function name: CancelAlarm
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
StatusType CancelAlarm ( AlarmType Alarm_ID)
{
	StatusType RetVal = E_OS_ID;
	
	if(MAX_NUM_OF_ALARMS > Alarm_ID)
	{
		Alarms[Alarm_ID].cycle = 0U;
		Alarms[Alarm_ID].increment = 0U;
		RetVal = E_OS_OK;
	}
	return RetVal;
}


/******************************************************************************
*       Function name: OS_Timer_Interrupts
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void OS_Timer_Interrupts(void)
{
	Update_OS_Alarms();
	ticks++;
}


/******************************************************************************
*       Function name: Update_OS_Alarms
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
static void Update_OS_Alarms(void)
{
	AlarmType OS_Alarms;

	for(OS_Alarms = 0U; OS_Alarms<MAX_NUM_OF_ALARMS; OS_Alarms++)
	{
		if((Alarms[OS_Alarms].cycle > 0U) || (Alarms[OS_Alarms].increment > 0U))
		{
			Alarms[OS_Alarms].increment--;
			if(0U == Alarms[OS_Alarms].increment)
			{
				ActivateTask (Alarms[OS_Alarms].TaskID);
				Alarms[OS_Alarms].increment = Alarms[OS_Alarms].cycle;
			}
		}
	}
}


/******************************************************************************
*       Function name: SetEvent
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
StatusType SetEvent ( TaskType TaskID )
{
    ActivateTask(TaskID);
    Is_Event_Activited++;
    return E_OS_OK;
}


/******************************************************************************
*       Function name: ClearEvent
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
StatusType ClearEvent ( TaskType TaskID )
{
    Tasks[TaskID].Task_State = TASK_SUSPENDED;
    Is_Event_Activited--;
    return E_OS_OK;
}


/******************************************************************************
*       Function name: TerminateEvent
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
StatusType TerminateEvent ( void )
{
    Tasks[Running_Task].Task_State = TASK_READY;
    return E_OS_OK;
}

/******************************************************************************
*                              End of file
******************************************************************************/
