#ifndef _EEPROM_H_
#define _EEPROM_H_
/******************************************************************************
*                    Texas Instruments Incorporated. All rights reserved
*******************************************************************************
*                    File: EEPROM.h
*             Description: EEPROM driver
*      Platform Dependent: no
*                   Notes:
*******************************************************************************/


/******************************************************************************
*
* Values returned by EEPROMInit.
*
*******************************************************************************/

/*
* This value may be returned from a call to EEPROMInit().  It indicates that
* no previous write operations were interrupted by a reset event and that the
* EEPROM peripheral is ready for use.
*/
#define EEPROM_INIT_OK      0

/*
* This value may be returned from a call to EEPROMInit().  It indicates that
* a previous data or protection write operation was interrupted by a reset
* event and that the EEPROM peripheral was unable to clean up after the
* problem.  This situation may be resolved with another reset or may be fatal
* depending upon the cause of the problem.  For example, if the voltage to
* the part is unstable, retrying once the voltage has stabilized may clear
* the error.
*/
#define EEPROM_INIT_ERROR   2

/******************************************************************************
*
* Error indicators returned by various EEPROM API calls.  These will be ORed
* together into the final return code.
*
******************************************************************************/

/*
* This return code bit indicates that an attempt was made to read from
* the EEPROM while a write operation was in progress.
*/
#define EEPROM_RC_WRBUSY            0x00000020

/*
* This return code bit indicates that an attempt was made to write a
* value but the destination permissions disallow write operations.  This
* may be due to the destination block being locked, access protection set
* to prohibit writes or an attempt to write a password when one is already
* written.
*/
#define EEPROM_RC_NOPERM            0x00000010

/*
* This return code bit indicates that the EEPROM programming state machine
* is currently copying to or from the internal copy buffer to make room for
* a newly written value.  It is provided as a status indicator and does not
* indicate an error.
*/
#define EEPROM_RC_WKCOPY            0x00000008

/*
* This return code bit indicates that the EEPROM programming state machine
* is currently erasing the internal copy buffer.  It is provided as a
* status indicator and does not indicate an error.
*/
#define EEPROM_RC_WKERASE           0x00000004

/*
* This return code bit indicates that the EEPROM programming state machine
* is currently working.  No new write operations should be attempted until
* this bit is clear.
*/
#define EEPROM_RC_WORKING           0x00000001

/******************************************************************************
*
* Values that can be passed to EEPROMBlockProtectSet() in the ui32Protect
* parameter, and returned by EEPROMBlockProtectGet().
*
******************************************************************************/

/*
* This bit may be ORed with the protection option passed to
* EEPROMBlockProtectSet() or returned from EEPROMBlockProtectGet().  It
* restricts EEPROM access to threads running in supervisor mode and prevents
* access to an EEPROM block when the CPU is in user mode.
*/
#define EEPROM_PROT_SUPERVISOR_ONLY 0x00000008

/*
* This value may be passed to EEPROMBlockProtectSet() or returned from
* EEPROMBlockProtectGet().  It indicates that the block should offer
* read/write access when no password is set or when a password is set and
* the block is unlocked, and read-only access when a password is set but
* the block is locked.
*/
#define EEPROM_PROT_RW_LRO_URW      0x00000000

/*
* This value may be passed to EEPROMBlockProtectSet() or returned from
* EEPROMBlockProtectGet().  It indicates that the block should offer neither
* read nor write access unless it is protected by a password and unlocked.
*/
#define EEPROM_PROT_NA_LNA_URW      0x00000001

/*
* This value may be passed to EEPROMBlockProtectSet() or returned from
* EEPROMBlockProtectGet().  It indicates that the block should offer
* read-only access when no password is set or when a password is set and the
* block is unlocked.  When a password is set and the block is locked, neither
* read nor write access is permitted.
*/
#define EEPROM_PROT_RO_LNA_URO      0x00000002

/******************************************************************************
*
* This value may be passed to EEPROMIntEnable() and EEPROMIntDisable() and is
* returned by EEPROMIntStatus() if an EEPROM interrupt is currently being
* signaled.
*
******************************************************************************/
#define EEPROM_INT_PROGRAM          0x00000004

/******************************************************************************
*
* Returns the EEPROM block number containing a given offset address.
*
* \param ui32Addr is the linear, byte address of the EEPROM location whose
* block number is to be returned.  This is a zero-based offset from the start
* of the EEPROM storage.
*
* This macro may be used to translate an EEPROM address offset into a
* block number suitable for use in any of the driver's block protection
* functions.  The address provided is expressed as a byte offset from the
* base of the EEPROM.
*
* \return Returns the zero-based block number which contains the passed
* address.
*
******************************************************************************/
#define EEPROMBlockFromAddr(ui32Addr) ((ui32Addr) >> 6)

/******************************************************************************
*
* Returns the offset address of the first word in an EEPROM block.
*
* \param ui32Block is the index of the EEPROM block whose first word address
* is to be returned.
*
* This macro may be used to determine the address of the first word in a
* given EEPROM block.  The address returned is expressed as a byte offset
* from the base of EEPROM storage.
*
* \return Returns the address of the first word in the given EEPROM block.
*
******************************************************************************/
#define EEPROMAddrFromBlock(ui32Block) ((ui32Block) << 6)

/***************************************************************************************************
*                              End of file
***************************************************************************************************/
#endif /* _EEPROM_H_ */
