/******************************************************************************
*                        EddeFlex All rights reserved
*******************************************************************************
*                    File: MCP2515.c
*             Description: MCP2515 driver
*      Platform Dependent: Std_Types.h
*                          SSI.h
*                          GPIO.h
*                 Version: 1
******************************************************************************/


/******************************************************************************
*                              Include section
******************************************************************************/
#include "Std_Types.h"
#include "hw_memmap.h"
#include "rom.h"
#include "GPIO.h"
#include "SSI.h"
#include "CAN.h"
#include "MCP2515.h"
#include "hw_MCP2515.h"

/******************************************************************************
*                               MACROS section
******************************************************************************/
#define MCP2515_CS_ACT              0x00U
#define MCP2515_CS_DACT             0xFFU
#define MCP2515_SSI_MODULE          SSI2_BASE
#define MCP2515_GPIO_MODULE         GPIO_PORTA_BASE
#define MCP2515_TXRX_BUFFER_SIZE    5U
#define MCP2515_MAX_DATA_LEN        8U
#define MCP2515_SIDH_INDEX          0U
#define MCP2515_SIDL_INDEX          1U
#define MCP2515_EID8_INDEX          2U
#define MCP2515_EID0_INDEX          3U
#define MCP2515_DLC_INDEX           4U
#define MCP2515_DLC_MASK            0x0FU
#define MCP2515_RTR_MASK            0x40U
#define MCP2515_SIDL_MASK           0xC0U
#define MCP2515_EIDI_MASK           0x03U
#define MCP2515_EIDF_MASK           0x08U
#define MCP2515_RX_BUFFER_FACTOR    4U
#define MCP2515_TX_BUFFER_FACTOR    2U
#define MCP2515_SIDH_SHIFT_VALUE    3U
#define MCP2515_SIDL_SHIFT_VALUE    5U
#define MCP2515_EIDI_SHIFT_VALUE    2U
#define MCP2515_BYTE_SHIFT_VALUE    8U
#define MCP2515_EXT_FLAG            0x08U

/******************************************************************************
*                                Datatypes
******************************************************************************/
typedef struct
{
    uint8 CFG1;
    uint8 CFG2;
    uint8 CFG3;
}MCP2515_BitRate_T;


/******************************************************************************
*                        Local Variables/Constants
******************************************************************************/
static uint8 MCP2515_CS;

static const MCP2515_BitRate_T MCP2515_BitRate[MCP2515_MAX_BIT_RATE_NUMBER] = 
{
    {MCP2515_16MHz_5kBPS_CFG1,MCP2515_16MHz_5kBPS_CFG2,MCP2515_16MHz_5kBPS_CFG3},
    {MCP2515_16MHz_10kBPS_CFG1,MCP2515_16MHz_10kBPS_CFG2,MCP2515_16MHz_10kBPS_CFG3},
    {MCP2515_16MHz_20kBPS_CFG1,MCP2515_16MHz_20kBPS_CFG2,MCP2515_16MHz_20kBPS_CFG3},
    {MCP2515_16MHz_31k25BPS_CFG3,MCP2515_16MHz_31k25BPS_CFG2,MCP2515_16MHz_31k25BPS_CFG3},
    {MCP2515_16MHz_40kBPS_CFG1,MCP2515_16MHz_40kBPS_CFG2,MCP2515_16MHz_40kBPS_CFG3},
    {MCP2515_16MHz_50kBPS_CFG1,MCP2515_16MHz_50kBPS_CFG2,MCP2515_16MHz_50kBPS_CFG3},
    {MCP2515_16MHz_80kBPS_CFG1,MCP2515_16MHz_80kBPS_CFG2,MCP2515_16MHz_80kBPS_CFG3},
    {MCP2515_16MHz_100kBPS_CFG1,MCP2515_16MHz_100kBPS_CFG2,MCP2515_16MHz_100kBPS_CFG3},
    {MCP2515_16MHz_125kBPS_CFG1,MCP2515_16MHz_125kBPS_CFG2,MCP2515_16MHz_125kBPS_CFG3},
    {MCP2515_16MHz_200kBPS_CFG1,MCP2515_16MHz_200kBPS_CFG2,MCP2515_16MHz_200kBPS_CFG3},
    {MCP2515_16MHz_250kBPS_CFG1,MCP2515_16MHz_250kBPS_CFG2,MCP2515_16MHz_250kBPS_CFG3},
    {MCP2515_16MHz_500kBPS_CFG1,MCP2515_16MHz_500kBPS_CFG2,MCP2515_16MHz_500kBPS_CFG3},
    {MCP2515_16MHz_1000kBPS_CFG1,MCP2515_16MHz_1000kBPS_CFG2,MCP2515_16MHz_1000kBPS_CFG3}
};
	

/******************************************************************************
*                              Local functions
******************************************************************************/
static uint8 SPI_transfer(uint8 u8Data);
static void  MCP2515_WriteReg(uint8 u8address, uint8 u8data);
static uint8 MCP2515_ReadReg(uint8 u8address);
static void  MCP2515_RequestSend(uint8 u8tx_Buff_id);
static void MCP2515_ClearBits(uint8 u8address, uint8 u8Bits);


/******************************************************************************
*                            Functions body
******************************************************************************/

/******************************************************************************
*       Function name: SPI_transfer
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
static uint8 SPI_transfer(uint8 u8Data)
{
    uint32 u32ret_Val;
    
    ROM_SSIDataPut(MCP2515_SSI_MODULE,u8Data);
    /* TODO timeout */
    while(TRUE == ROM_SSIBusy(MCP2515_SSI_MODULE));
    ROM_SSIDataGet(MCP2515_SSI_MODULE,&u32ret_Val);
    return u32ret_Val;
}


/******************************************************************************
*       Function name: MCP2515_WriteReg
*         Description: write register
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
static void MCP2515_WriteReg(uint8 u8address, uint8 u8data)
{
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_ACT);
    (void)SPI_transfer(MCP2515_WRITE_CMD);
    (void)SPI_transfer(u8address);
    (void)SPI_transfer(u8data);
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_DACT);    
}


/******************************************************************************
*       Function name: MCP2515_ReadReg
*         Description: read register
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
static uint8 MCP2515_ReadReg(uint8 u8address)
{
    uint8 ret_val;
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_ACT);
    (void)SPI_transfer(MCP2515_READ_CMD);
    (void)SPI_transfer(u8address);
    ret_val = SPI_transfer(MCP2515_DUMMY_DATA);    
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_DACT);
    return ret_val;
}


/******************************************************************************
*       Function name: MCP2515_RequestSend
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
static void MCP2515_RequestSend(uint8 tx_Buff_id)
{
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_ACT);
    (void)SPI_transfer(MCP2515_RTS_CMD | (1U << tx_Buff_id));
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_DACT);
}


/******************************************************************************
*       Function name: MCP2515_ClearBits
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
static void MCP2515_ClearBits(uint8 u8address, uint8 u8Bits)
{
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_ACT);
    (void)SPI_transfer(MCP2515_BIT_MODIFY_CMD);
    (void)SPI_transfer(u8address);
    (void)SPI_transfer(u8Bits);
    (void)SPI_transfer(MCP2515_BIT_CLEAR);
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_DACT);
}


/******************************************************************************
*       Function name: MCP2515_SetBits
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
static void MCP2515_SetBits(uint8 u8address, uint8 u8Bits)
{
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_ACT);
    (void)SPI_transfer(MCP2515_BIT_MODIFY_CMD);
    (void)SPI_transfer(u8address);
    (void)SPI_transfer(u8Bits);
    (void)SPI_transfer(MCP2515_BIT_SET);
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_DACT);
}


/******************************************************************************
*       Function name: MCP2515_Rest
*         Description: rest MCP2515
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void MCP2515_SW_Reset(uint8 u8CSPin)
{
    uint8 u8MCP_Status;
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_ACT);
    (void)SPI_transfer(MCP2515_REST_CMD);
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_DACT);
    /* TODO Timeout */
    do
    {
        u8MCP_Status = MCP2515_ReadReg(MCP2515_CANSTAT);
    }while(MCP2515_MODE_CFG != u8MCP_Status);
}


/******************************************************************************
*       Function name: MCP2515_Init
*         Description: initialize MCP2515
*     Parameters (IN): bit_rate
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void MCP2515_Init(uint8 u8CSPin,uint8 u8BitRate)
{
	MCP2515_CS = u8CSPin;

    MCP2515_SW_Reset(u8CSPin);

    MCP2515_WriteReg(MCP2515_CNF1,MCP2515_BitRate[u8BitRate].CFG1);
    MCP2515_WriteReg(MCP2515_CNF2,MCP2515_BitRate[u8BitRate].CFG2);
    MCP2515_WriteReg(MCP2515_CNF3,MCP2515_BitRate[u8BitRate].CFG3);
    	
	/* TODO Remove from here */
	MCP2515_WriteReg(MCP2515_RXB0CTRL, 0x64);
	MCP2515_WriteReg(MCP2515_RXB1CTRL, 0x60);
}


/******************************************************************************
*       Function name: MCP2515_Enable
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void  MCP2515_Enable(uint8 u8CSPin)
{
    MCP2515_CS = u8CSPin;
    MCP2515_WriteReg(MCP2515_CANCTRL,MCP2515_MODE_NORMAL);
}


/******************************************************************************
*       Function name: MCP2515_Disable
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void  MCP2515_Disable(uint8 u8CSPin)
{
    MCP2515_CS = u8CSPin;
    MCP2515_WriteReg(MCP2515_CANCTRL,MCP2515_MODE_SLEEP);
}


/******************************************************************************
*       Function name: MCP2515_Msg_Send
*         Description: send CAN Msg
*     Parameters (IN): Msg
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: - 
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void  MCP2515_CANMessageSet(uint8 u8CSPin, uint8 u8BuffId, strCANMsgObject *psMsgObject)
{
    uint8 counter;
    uint8 u8TX_Write_CMD;
    uint8 Reg_Buff[MCP2515_TXRX_BUFFER_SIZE];
    uint32 u32MsgID;

    MCP2515_CS = u8CSPin;

    u8TX_Write_CMD = ((u8BuffId * MCP2515_TX_BUFFER_FACTOR) | MCP2515_WRITE_TXB_CMD);

    u32MsgID = psMsgObject->ui32MsgID;

    if(MSG_OBJ_NO_FLAGS == (psMsgObject->ui32Flags & MSG_OBJ_EXTENDED_ID))
    {
        Reg_Buff[MCP2515_SIDH_INDEX] = (uint8)(u32MsgID >> MCP2515_SIDH_SHIFT_VALUE);
        Reg_Buff[MCP2515_SIDL_INDEX] = (uint8)(u32MsgID << MCP2515_SIDL_SHIFT_VALUE);
        Reg_Buff[MCP2515_EID8_INDEX] = MCP2515_DUMMY_DATA;
        Reg_Buff[MCP2515_EID0_INDEX] = MCP2515_DUMMY_DATA;
    }
    else
    {
        Reg_Buff[MCP2515_EID0_INDEX] = (uint8)u32MsgID;
        u32MsgID = u32MsgID >> MCP2515_BYTE_SHIFT_VALUE;
        Reg_Buff[MCP2515_EID8_INDEX] = (uint8)u32MsgID;
        u32MsgID = u32MsgID >> MCP2515_BYTE_SHIFT_VALUE;
        Reg_Buff[MCP2515_SIDL_INDEX] = (uint8)(u32MsgID & MCP2515_EIDI_MASK);
        Reg_Buff[MCP2515_SIDH_INDEX] = (uint8)(u32MsgID >> MCP2515_SIDL_SHIFT_VALUE);
        u32MsgID = u32MsgID << MCP2515_SIDH_SHIFT_VALUE;
        Reg_Buff[MCP2515_SIDL_INDEX] = Reg_Buff[MCP2515_SIDL_INDEX] | ((uint8) (u32MsgID & MCP2515_SIDL_MASK));
        Reg_Buff[MCP2515_SIDL_INDEX] = Reg_Buff[MCP2515_SIDL_INDEX] | MCP2515_EXT_FLAG;
    }
    Reg_Buff[MCP2515_DLC_INDEX] = (uint8)(psMsgObject->ui32MsgLen & MCP2515_DLC_MASK);
    Reg_Buff[MCP2515_DLC_INDEX] = Reg_Buff[MCP2515_DLC_INDEX] | ((uint8)(psMsgObject->ui32Flags & MSG_OBJ_REMOTE_FRAME));

    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_ACT);
    (void)SPI_transfer(u8TX_Write_CMD);
    
    for(counter = 0U; counter < MCP2515_TXRX_BUFFER_SIZE; counter++)
    {
        (void)SPI_transfer(Reg_Buff[counter]);
    }
    for(counter = 0U; counter < (psMsgObject->ui32MsgLen); counter++)
    {
        (void)SPI_transfer((psMsgObject->pui8MsgData)[counter]);
    }
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_DACT);
    MCP2515_RequestSend(u8BuffId);
}
/**************************************************************************************************/


/******************************************************************************
*       Function name: MCP2515_Msg_Receive
*         Description: receive CAN Msg
*     Parameters (IN): Msg
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: - 
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void  MCP2515_CANMessageGet(uint8 u8CSPin, uint8 u8BuffId, strCANMsgObject *psMsgObject)
{
	uint8 counter;
	uint8 Reg_Buff[MCP2515_TXRX_BUFFER_SIZE];

	u8BuffId = (MCP2515_Read_RXB_CMD | (u8BuffId * MCP2515_RX_BUFFER_FACTOR));
	ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_ACT);
	(void)SPI_transfer(u8BuffId);

	for(counter = 0U; counter < MCP2515_TXRX_BUFFER_SIZE; counter++)
	{
	    Reg_Buff[counter] = SPI_transfer(MCP2515_DUMMY_DATA);
	}
	for(counter = 0U; counter< MCP2515_MAX_DATA_LEN; counter++)
	{
	    (psMsgObject->pui8MsgData)[counter] = SPI_transfer(MCP2515_DUMMY_DATA);
	}

	ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, MCP2515_CS, MCP2515_CS_DACT);

	psMsgObject->ui32MsgLen = Reg_Buff[MCP2515_DLC_INDEX] & MCP2515_DLC_MASK;

	psMsgObject->ui32MsgID = Reg_Buff[MCP2515_SIDH_INDEX];
	psMsgObject->ui32MsgID = (psMsgObject->ui32MsgID << MCP2515_SIDH_SHIFT_VALUE);
	psMsgObject->ui32MsgID = (psMsgObject->ui32MsgID | (Reg_Buff[MCP2515_SIDL_INDEX] >> MCP2515_SIDL_SHIFT_VALUE));

	psMsgObject->ui32Flags = (Reg_Buff[MCP2515_DLC_INDEX] & MCP2515_RTR_MASK);

	if(MSG_OBJ_NO_FLAGS != (Reg_Buff[MCP2515_SIDL_INDEX] & MCP2515_EIDF_MASK))
	{
	    psMsgObject->ui32Flags |= MSG_OBJ_EXTENDED_ID;

		psMsgObject->ui32MsgID = (psMsgObject->ui32MsgID << MCP2515_EIDI_SHIFT_VALUE);
		psMsgObject->ui32MsgID = (psMsgObject->ui32MsgID | (Reg_Buff[MCP2515_SIDL_INDEX] & MCP2515_EIDI_MASK));
		psMsgObject->ui32MsgID = (psMsgObject->ui32MsgID << MCP2515_BYTE_SHIFT_VALUE);
		psMsgObject->ui32MsgID = (psMsgObject->ui32MsgID | Reg_Buff[MCP2515_EID8_INDEX]);
		psMsgObject->ui32MsgID = (psMsgObject->ui32MsgID << MCP2515_BYTE_SHIFT_VALUE);
		psMsgObject->ui32MsgID = (psMsgObject->ui32MsgID | Reg_Buff[MCP2515_EID0_INDEX]);
	}

}


/******************************************************************************
*       Function name: MCP2515_CANStatusGet
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
uint8 MCP2515_CANStatusGet(uint8 u8CSPin)
{
    uint8 u8retval;
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, u8CSPin, MCP2515_CS_ACT);
    (void)SPI_transfer(MCP2515_READ_STATUS_CMD);
    u8retval = SPI_transfer(MCP2515_DUMMY_DATA);
    ROM_GPIOPinWrite(MCP2515_GPIO_MODULE, u8CSPin, MCP2515_CS_DACT);
    return u8retval;
}


/******************************************************************************
*       Function name: MCP2515_CANIntEnable
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void  MCP2515_CANIntEnable(uint8 u8CSPin, uint8 u8IntFlags)
{
    MCP2515_CS = u8CSPin;
    MCP2515_SetBits(MCP2515_CANINTE, u8IntFlags);
}


/******************************************************************************
*       Function name: MCP2515_CANIntDisable
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void  MCP2515_CANIntDisable(uint8 u8CSPin, uint8 u8IntFlags)
{
    MCP2515_CS = u8CSPin;
    MCP2515_ClearBits(MCP2515_CANINTE, u8IntFlags);
}


/******************************************************************************
*       Function name: MCP2515_CANIntClear
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void  MCP2515_CANIntClear(uint8 u8CSPin, uint8 u8IntFlags)
{
    MCP2515_CS = u8CSPin;
    MCP2515_ClearBits(MCP2515_CANINTF, u8IntFlags);
}


/******************************************************************************
*       Function name: MCP2515_CANIntStatus
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: <global variables accessed in function>
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
uint8 MCP2515_CANIntStatus(uint8 u8CSPin)
{
    uint8 u8retval;
    uint8 u8IntMask;
    MCP2515_CS = u8CSPin;
    u8IntMask = MCP2515_ReadReg(MCP2515_CANINTE);
    u8retval = MCP2515_ReadReg(MCP2515_CANINTF);
    u8retval = (u8retval & u8IntMask);
    return u8retval;
}


/******************************************************************************
*                             File History
******************************************************************************
* 3/4/2017    YAli    -Initial Version.
******************************************************************************
******************************************************************************
*                              End of file
******************************************************************************/
