/******************************************************************************
*                       EddeFlex All rights reserved
*******************************************************************************
*                    File: main.c
*             Description: main file
*      Platform Dependent: no
*                   Notes:
******************************************************************************/


/******************************************************************************
*                               Includes
******************************************************************************/
#include "std_types.h"
#include "hw_memmap.h"
#include "sysctl.h"
#include "rom.h"
#include "pin_map.h"
#include "EddeOS.h"
#include "GPIO.h"
#include "UART.h"
#include "SSI.h"
#include "CAN.h"
#include "MCP2515.h"


/******************************************************************************
*                                MACROS
******************************************************************************/
#define CAN_BAUDRATE_500K            500
#define RXCAN_Msg_Obj                3
#define UART0_BaudRate               115200
#define UART_CANData_Msg             "\nCANData : "
#define UART_CANID_Msg               "CANID : 0x"
#define UART_New_Line_Char           '\n'
#define UART_Hex_Header              " 0x"
#define UART_Draw_Line               "*****************************************************************************\n"
#define UART_CAN0_ID                 "RX from CAN0\n"
#define UART_CAN1_ID                 "RX from CAN1\n"
#define UART_Start_Connection_Msg    "Start connection:\n"
#define CAN0_ID                      0U
#define MCP2515_SSI_Max_ClockRate    10000000UL
#define MCP2515_SSI_Data_Width       8U
#define MCP2515_CS12                 GPIO_PIN_3

/******************************************************************************
*                                Constants
******************************************************************************/
const uint8* u8HexValues = "0123456789ABCDEF";


/******************************************************************************
*                                Variables
******************************************************************************/
extern strCANMsgObject RXCAN_Msg;


/******************************************************************************
*       Function name: UART0_SendString_Blocking
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void UART0_SendString_Blocking(const uint8* u8Data,const uint8 u8Data_Len)
{
    uint8 u8counter;

    for(u8counter = 0; u8counter < u8Data_Len; u8counter++)
    {
        ROM_UARTCharPut(UART0_BASE,u8Data[u8counter]);
    }
}


/******************************************************************************
*       Function name: UART_Write_CAN_Msg
*         Description: -
*     Parameters (IN): u32MsgID
*                      u8MsgData
*                      u8MsgLen
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void UART0_Write_CAN_Msg(const uint8 u8CANCntID,uint32 u32MsgID, uint8* u8MsgData, uint8 u8MsgLen)
{
    uint32 u32temp;
    uint8  ConvVal;
    sint8  counter;

    UART0_SendString_Blocking(UART_Draw_Line, (sizeof(UART_Draw_Line)-1));
    if(CAN0_ID == u8CANCntID)
    {
        UART0_SendString_Blocking(UART_CAN0_ID,(sizeof(UART_CAN0_ID)-1));
    }
    else
    {
        UART0_SendString_Blocking(UART_CAN1_ID,(sizeof(UART_CAN1_ID)-1));
    }
    UART0_SendString_Blocking(UART_CANID_Msg,(sizeof(UART_CANID_Msg)-1));

    for(counter = 7; counter >= 0; counter--)
    {
        u32temp = (u32MsgID >> (counter*4)) & 0xFU;
        ConvVal = u8HexValues[u32temp];
        ROM_UARTCharPut(UART0_BASE,ConvVal);
    }
    UART0_SendString_Blocking(UART_CANData_Msg,(sizeof(UART_CANData_Msg)-1));
    for(counter = 0; counter < u8MsgLen; counter++)
    {
        UART0_SendString_Blocking(UART_Hex_Header,(sizeof(UART_Hex_Header)-1));
        u32temp = u8MsgData[counter] >> 4;
        ConvVal = u8HexValues[u32temp];
        ROM_UARTCharPut(UART0_BASE,ConvVal);
        u32temp = u8MsgData[counter] & 0xFU;
        ConvVal = u8HexValues[u32temp];
        ROM_UARTCharPut(UART0_BASE,ConvVal);
    }
    ROM_UARTCharPut(UART0_BASE,UART_New_Line_Char);
    UART0_SendString_Blocking(UART_Draw_Line, (sizeof(UART_Draw_Line)-1));
}


/******************************************************************************
*       Function name: UART0_Init
*         Description: -
*     Parameters (IN): u32Baudrate : baudrate of UART0
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void UART0_Init(uint32 u32Baudrate)
{
    /* Enable clock to GPIO A */
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);
    /* Configure GPIO A pin 0 and 1 to be UART TX and RX  */
    ROM_GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0|GPIO_PIN_1);
    ROM_GPIOPinConfigure(GPIO_PA0_U0RX);
    ROM_GPIOPinConfigure(GPIO_PA1_U0TX);
    /* Enable clock to UART0 */
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);
    /* select UART clock source to be main clock */
    ROM_UARTClockSourceSet(UART0_BASE,UART_CLOCK_SYSTEM);
    /* Config UART0 */
    ROM_UARTConfigSetExpClk(UART0_BASE,ROM_SysCtlClockGet(),u32Baudrate,
                            UART_CONFIG_WLEN_8|UART_CONFIG_STOP_ONE|UART_CONFIG_PAR_NONE);
    /* Enable UART0 */
    ROM_UARTEnable(UART0_BASE);
}


/******************************************************************************
*       Function name: CANBp_MCP2515_Init
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void EddeExternalCANBp_Init(uint8 u8CSPin, uint16 u16BaudRate)
{
    /* Enable GPIOA clock for CS pin */
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);
    /* Config CS pin as output */
    ROM_GPIOPinTypeGPIOOutput(GPIO_PORTA_BASE,u8CSPin);

    /* Enable GPIOB clock for SSI2 pins */
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOB);
    /* Config SSI2 pins */
    ROM_GPIOPinTypeSSI(GPIO_PORTB_BASE,GPIO_PIN_4|GPIO_PIN_6|GPIO_PIN_7);
    ROM_GPIOPinConfigure(GPIO_PB4_SSI2CLK);
    ROM_GPIOPinConfigure(GPIO_PB6_SSI2RX);
    ROM_GPIOPinConfigure(GPIO_PB7_SSI2TX);

    /* Enable clock to SSI2 module */
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_SSI2);
    /* select main clock to be SSI2 clock */
    ROM_SSIClockSourceSet(SSI2_BASE,SSI_CLOCK_SYSTEM);
    /* Config SSI2 */
    ROM_SSIConfigSetExpClk(SSI2_BASE,ROM_SysCtlClockGet(),SSI_FRF_MOTO_MODE_0,
                           SSI_MODE_MASTER,MCP2515_SSI_Max_ClockRate,MCP2515_SSI_Data_Width);
    /* Enable SSI2 */
    ROM_SSIEnable(SSI2_BASE);

    /* Init MCP2515 */
    MCP2515_Init(u8CSPin,u16BaudRate);
    /* Enable MCP2515 */
    MCP2515_Enable(u8CSPin);
}


/******************************************************************************
*       Function name: EddeNativeCANBp_Init
*         Description: Init CAN TX, RX and controller
*     Parameters (IN): u16BaudRate : baud rate in KHz
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void EddeNativeCANBp_Init(uint16 u16BaudRate)
{
    uint32 u32AbsBaudRate = u16BaudRate * 1000U;

    /* Enable clock to GPIO E */
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOE);
    /* Configure GPIO E pin 4 and 5 to be CAN TX and RX  */
    ROM_GPIOPinTypeCAN(GPIO_PORTE_BASE, GPIO_PIN_4|GPIO_PIN_5);
    ROM_GPIOPinConfigure(GPIO_PE4_CAN0RX);
    ROM_GPIOPinConfigure(GPIO_PE5_CAN0TX);
    /* Enable clock to CAN0 */
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_CAN0);
    /* Set baudrate of CAN0 */
    ROM_CANBitRateSet(CAN0_BASE,ROM_SysCtlClockGet(),u32AbsBaudRate);
    /* Enable CAN0 Controller */
    ROM_CANEnable(CAN0_BASE);
}


/******************************************************************************
*       Function name: main
*         Description: -
*     Parameters (IN): -
*    Parameters (OUT): -
* Parameters (IN/OUT): -
*        Return value: -
*    Global variables: -
*             Remarks: <any other details including if this function disables interrupt>
******************************************************************************/
void main(void)
{
    /* Set main clock to 80MHz */
    ROM_SysCtlClockSet(PLL_80MHz);
    /* Enable clock to GPIO F */
    ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOF);
    /* Config LED output pins */
    ROM_GPIOPinTypeGPIOOutput(GPIO_PORTF_BASE,GPIO_PIN_1|GPIO_PIN_2);
    /* Init native CAN controller */
    EddeNativeCANBp_Init(CAN_BAUDRATE_500K);
    /* Set RX object in msgs objects  */
    ROM_CANMessageSet(CAN0_BASE,RXCAN_Msg_Obj,&RXCAN_Msg,MSG_OBJ_TYPE_RX);
    /* Init external CAN controller */
    EddeExternalCANBp_Init(MCP2515_CS12,MCP2515_BitRate_500kBPS);
    /* Init UART0 */
    UART0_Init(UART0_BaudRate);
    /* Send Start indication */
    UART0_SendString_Blocking(UART_Start_Connection_Msg,(sizeof(UART_Start_Connection_Msg)-1));
    /* Init EddeOS */
    EddeOS_Init();
    /* Start OS */
	EddeOS_StartOS(OS_NORMAL_MODE);
}

/******************************************************************************
*                              End of file
*******************************************************************************/
